import crypto from "crypto";
import fs from "fs";
import axios from "axios";

function parseJson(prefix, obj, result) {
    if (typeof obj === "object" && !Array.isArray(obj)) {
        for (const key in obj) {
            const newPrefix = prefix ? `${prefix}:${key}` : key;
            parseJson(newPrefix, obj[key], result);
        }
    } else {
        result.push(`${prefix}:${obj}`);
    }
}

function normalizeMessage(payload) {
    let result = [];
    parseJson("", payload, result);
    result.sort();
    return result.join(";");
}

async function makeRequest(projectId, privateKeyFilePath, url, payload) {
    const privateKey = fs.readFileSync(privateKeyFilePath, "utf8");


    const normalizedMessage = normalizeMessage(payload);

    const timestamp = Math.floor(Date.now() / 1000);

    const base64EncodedMessage =
        Buffer.from(normalizedMessage).toString("base64");
    const message = `${base64EncodedMessage}${timestamp}`;

    const signer = crypto.createSign("RSA-SHA256");
    signer.update(message);
    signer.end();

    const signature = signer.sign(privateKey);
    const base64Sign = signature
        .toString("base64")
        .replace(/\+/g, "-")
        .replace(/\//g, "_");

    const publicKey = crypto
        .createPublicKey(privateKey)
        .export({type: "pkcs1", format: "pem"});
    const apiKey = Buffer.from(publicKey)
        .toString("base64")
        .replace(/\+/g, "-")
        .replace(/\//g, "_")
        .replace(/=+$/, "");

    const headers = {
        "content-type": "application/json",
        "x-access-token": apiKey,
        "x-access-signature": base64Sign,
        "x-access-merchant-id": projectId,
        "x-access-timestamp": timestamp.toString(),
    };

    try {
        const response = await axios.post(url, payload, {headers});
        console.log("Response status:", response.status);
        console.log("Response data:", response.data);
        return response
    } catch (error) {
        console.error(
            "Error:",
            error.response ? error.response.data : error.message
        );
    }
}

export { makeRequest };